# -*- coding: utf-8 -*-

import arcpy


class Toolbox(object):
    def __init__(self):
        """Define the toolbox (the name of the toolbox is the name of the
        .pyt file)."""
        self.label = "Toolbox"
        self.alias = "toolbox"

        # List of tool classes associated with this toolbox
        self.tools = [Tool]


class Tool(object):
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "SHRA Screening Tool"
        self.description = ""
        self.canRunInBackground = False

    def getParameterInfo(self):
       # First parameter
        param0 = arcpy.Parameter(
            displayName="Injection Well",
            name="Injec",
            datatype="GPPoint",
            parameterType="Required",
            direction="Input",
            multiValue=True)
        # Second parameter
        param1 = arcpy.Parameter(
            displayName="Production Well",
            name="Prod",
            datatype="GPPoint",
            parameterType="Required",
            direction="Input",
            multiValue=False)
         # Third parameter
        param2 = arcpy.Parameter(
            displayName="Workspace",
            name="location",
            datatype="DEWorkspace",
            parameterType="Optional",
            direction="Input",
            multiValue=False)
        param2.value = arcpy.env.workspace
        # Fourth parameter
        param3 = arcpy.Parameter(
            displayName="Spatial Reference System",
            name="srs",
            datatype="GPSpatialReference",
            parameterType="Required",
            direction="Input",
            multiValue=False)
        param3.value = arcpy.SpatialReference('RD_New')
        param4 = arcpy.Parameter(
            displayName="Output Map Name",
            name="mapname",
            datatype="GPString",
            parameterType="Optional",
            direction="Input",
            multiValue=False)
        param4.value = 'SHRA Map'
        params=[param0,param1,param2,param3,param4]
        return params

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self,params):
        if params[1].value is not None:
            prdw=params[1].valueAsText
            coords = str(prdw).replace(',','.').split(' ')
            x = float(coords[0])
            y = float(coords[1])
            if x > 8 or x < 2 or y > 54 or y < 50: 
                params[1].value= None
                arcpy.AddMessage('outside boundaries')
        if params[0].value is not None:
            injw=params[0].values
        # creating points for injection wells
            for iw in injw:
                coords = str(iw).replace(',','.').split(' ')
                x = float(coords[0])
                y = float(coords[1])
                if x > 8 or x < 2 or y > 54 or y < 50: 
                    params[0].value= None
                    arcpy.AddMessage('outside boundaries')
        if params[2].valueAsText is None:
            params[2].value= arcpy.env.workspace
            arcpy.AddMessage('No workspace set, so defaulting to default workspace')
        if params[2].valueAsText.endswith(".gdb") == False:
            params[2].value= arcpy.env.workspace
            arcpy.AddMessage('No gdb workspace set, so defaulting to default workspace')
        if params[4].valueAsText is None:
            params[2].value= 'SHRA Map'
            arcpy.AddMessage('No mapname set, so defaulting to default Name')         
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        return

        
    def mapbuilder(self,mapname,outFC,analist,mess):        
        aprx = arcpy.mp.ArcGISProject("Current")
        for Maps in aprx.listMaps():
            if Maps.name==mapname:
                for things in analist:
                    loc=os.path.join(outFC,things)
                    locin=loc+'_intersected'
                    desc = arcpy.Describe(locin)
                    if desc.shapeType == 'Polygon':

                        layer=Maps.addDataFromPath(loc)
                        sym=layer.symbology
                        sym.renderer.symbol.color = {'RGB' : [0, 115, 76, 20]}
                        sym.renderer.symbol.outlineColor = {'RGB' : [0, 0, 0, 20]}
                        sym.renderer.symbol.outlineSize = 1
                        sym.renderer.symbol.size =2
                        layer.symbology = sym


                        locinlayer=Maps.addDataFromPath(locin)
                        locinsym=locinlayer.symbology
                        locinsym.renderer.symbol.listSymbolsFromGallery("red")[0]
                        locinsym.renderer.symbol.color = {'RGB' : [255, 0, 0, 80]}
                        locinsym.renderer.symbol.outlineColor = {'RGB' : [0, 0, 0, 80]}
                        locinsym.renderer.symbol.outlineSize = 1
                        locinsym.renderer.symbol.size = 2          
                        locinlayer.symbology = locinsym 


                    elif desc.shapeType == 'Point':
                        layer=Maps.addDataFromPath(loc)
                        sym=layer.symbology
                        sym.renderer.symbol.color = {'RGB' : [0, 115, 76, 20]}
                        sym.renderer.symbol.outlineColor = {'RGB' : [0, 0, 0, 20]}
                        sym.renderer.symbol.outlineSize = 1
                        sym.renderer.symbol.size =10
                        layer.symbology = sym


                        locinlayer=Maps.addDataFromPath(locin)
                        locinsym=locinlayer.symbology
                        locinsym.renderer.symbol.color = {'RGB' : [255, 0, 0, 80]}
                        locinsym.renderer.symbol.outlineColor = {'RGB' : [0, 0, 0, 80]}
                        locinsym.renderer.symbol.outlineSize = 1
                        locinsym.renderer.symbol.size = 10          
                        locinlayer.symbology = locinsym


                
                layer=Maps.addDataFromPath(os.path.join(outFC,'WellsAOI'))
                sym=layer.symbology
                sym.renderer.type == "SimpleRenderer"      
                sym.renderer.symbol = sym.renderer.symbol.listSymbolsFromGallery("Dash")[0]
                layer.symbology = sym  

                layer=Maps.addDataFromPath(os.path.join(outFC,'Wellsbuffered'))
                sym=layer.symbology
                sym.updateRenderer('UniqueValueRenderer')
                sym.renderer.fields = ['Welltype']
                sym.renderer.addValues({"Welltype" : ["production", "injection"]})
                for grp in sym.renderer.groups:
                    for itm in grp.items:
                        myVal = itm.values[0][0]

                        if myVal == "injection":
                            print("num 1")
                            itm.symbol.color = {"RGB": [0, 20, 237, 50]}      # Blue
                            itm.symbol.size = 2
                            itm.label = str(myVal)+ ' AOI'

                        elif myVal == "production":
                            print("num 3")
                            itm.symbol.color = {"RGB": [219, 0, 200, 50]}      # green
                            itm.symbol.size = 2
                            itm.label = str(myVal)+ ' AOI'

                layer.symbology = sym

                layer=Maps.addDataFromPath(os.path.join(outFC,'Wells'))
                sym=layer.symbology
                sym.updateRenderer('UniqueValueRenderer')
                sym.renderer.fields = ['Welltype']
                sym.renderer.addValues({"Welltype" : ["production", "injection"]})
                for grp in sym.renderer.groups:
                    for itm in grp.items:
                        myVal = itm.values[0][0]

                        if myVal == "injection":
                            print("num 1")
                            itm.symbol.color = {"RGB": [0, 20, 237, 50]}      # Blue
                            itm.symbol.size = 5
                            itm.label = str(myVal)+ ' Well'

                        elif myVal == "production":
                            print("num 3")
                            itm.symbol.color = {"RGB": [219, 0, 200, 50]}      # purple
                            itm.symbol.size = 5
                            itm.label = str(myVal)+ ' Well'
                layer.symbology = sym
                if len(analist) ==0:
                    arcpy.AddMessage('Geen overlap gevonden tussen '.format(loc=things) )
                count=0
                for things in analist:
                    loc=os.path.join(outFC,things)
                    locin=loc+'_intersected'
                    desc = arcpy.Describe(locin)             
                    if arcpy.management.GetCount(locin)[0] != '0':
                        for key,value in sorted(mess.items()):
                            for values in value:
                                if things == values.split(':')[-1]:
                                    count+=1
                                    arcpy.AddMessage('De Area Of Influence van dit geothermie systeem overlapt met {loc}, kijk in de de SHRA documentatie voor de vervolgstappen'.format(loc=things) )



                    else:
                        self.deletewrapper(loc)
                        self.deletewrapper(locin)
                if count ==0:
                    arcpy.AddMessage('Geen overlap gevonden met de relevantie lagen uit de webservice en de Area of influence van het geothermie systeem')

    def wellscreator(self,x,y,sr):
        point = arcpy.Point()
        point.X = float(x)
        point.Y = float(y)
        pointGeometry = arcpy.PointGeometry(point, sr)
        return (pointGeometry)

    def featurecreator(self,wells,outfc,sr,ft,name):
        arcpy.CreateFeatureclass_management(outfc,  
                                        name, 
                                        ft,spatial_reference= sr)
        arcpy.management.AddField(name, "Welltype", "Text")
        arcpy.management.AddField(name, "Count", "Short")
        fields = ["Welltype","Count","SHAPE@"]
        cur = arcpy.da.InsertCursor(os.path.join(outfc,name), fields)
        count=0
        for wellsx in wells: 
            cur.insertRow([wellsx[1],count,wellsx[0]])
            count+=1
        del cur

        
    def buffer(self,wells,inputfile,outFC):
        iw=[]
        count=0
        for wellsx in wells:
                         
            if wellsx[1]=='production':
                field = "Welltype"  
                whereClause = """"{}" = '{}'""".format(field,wellsx[1])
                arcpy.analysis.Select(inputfile, r"memory\Buffers", whereClause)           
                pw=arcpy.analysis.Buffer(r"memory\Buffers", r"memory\productionwellbuffer", '300 Meter', 'Full','Round')
            if wellsx[1]=='injection':
                field = "Count"  
                whereClause = """"{}" = {}""".format(field,count)
                arcpy.analysis.Select(inputfile, r"memory\Buffers", whereClause)
                distance = str(0.7*wellsx[-1])+' Meter'
                outfile= r"memory\injectionwellbuffer"+str(count)
                iw.append(arcpy.analysis.Buffer(r"memory\Buffers",outfile, distance, 'Full','Round'))

            count+=1
        pwout=[]
        for row in arcpy.da.SearchCursor(pw, ["SHAPE@"]):
            pwout.append([row[0],'production'])
        for ijw in iw:
            for row in arcpy.da.SearchCursor(ijw, ["SHAPE@"]):
                pwout.append([row[0],'injection'])
        wells=pwout
        
        
        return(wells)

            
    def errorhandler(self,code):
        if code == 1000:
            message='The only valid coordinate systems for this tool are GCS_WGS_1984 (4326/3857), ED_1950_UTM_Zone_31N (23031) or RD_NEW (28992)'
        if code == 1001:
            message='The distance of the wells is too big, check the coordinates'
        arcpy.AddError(message)
        sys.exit(0)

    def convexhull(self,wells,outfc,env=True):
        arcpy.MinimumBoundingGeometry_management(wells, outfc, "CONVEX_HULL", group_option="ALL")


    def boundingbox(self,convexhul,outFC):
        fc=arcpy.analysis.Buffer(convexhul, os.path.join(outFC,'Searchbox'), '10000 Meter', 'Full','Round')
        out_coordinate_system = arcpy.SpatialReference('WGS_1984_Web_Mercator_Auxiliary_Sphere')
        arcpy.management.Project(os.path.join(outFC,'Searchbox'), os.path.join(outFC,'Searchboxrepro'), out_coordinate_system)
        descFC = arcpy.Describe(os.path.join(outFC,'Searchboxrepro'))
        bbox=str(descFC.extent.XMin)+','+str(descFC.extent.YMin)+','+str(descFC.extent.XMax)+','+str(descFC.extent.YMax)
        self.deletewrapper(os.path.join(outFC,'Searchboxrepro'))
        self.deletewrapper(os.path.join(outFC,'Searchbox'))
        return(bbox)

    def serviceloader(self,bbox,outFC):
        import requests
        import xml.etree.ElementTree as ET
        import urllib
        import json
        wfslink="https://dservices1.arcgis.com/s0nkxJsg6Q24YYHt/arcgis/services/SHRA_geothermie_overzichtskaart/WFSServer"

        getcab= {'service':'wfs','request':'getcapabilities'}
        resp = requests.get(wfslink, params=getcab)
        root = ET.fromstring(resp.content)
        count=0
        analist=[]
        for x in root.iter('{http://www.opengis.net/wfs/2.0}Title'):

            WFS_FeatureType = x.text
            params='<ogc:Filter><ogc:BBOX><ogc:PropertyName>Shape</ogc:PropertyName><gml:Box%20srsName=\"urn:x-ogc:def:crs:EPSG:3857\"><gml:coordinates>'+bbox+'</gml:coordinates></gml:Box></ogc:BBOX></ogc:Filter>'

            qn='SHRA_geothermie_overzichtskaart:'+x.text
            getfeat= {'service':'wfs'
                     ,'request':'GetFeature'
                     ,'version':'2.0.0'
                     ,'typename':qn
                     ,'outputFormat':'ESRIGEOJSON'
                     ,'filter': params
                     
                     }

            ft = requests.get(wfslink, params=getfeat)
            #arcpy.AddMessage(ft.request.url)
            if ft.text == '':
                pass
            else:
                data =ft.json()

                if len(data['features']) > 0:
                    arcpy.AddMessage('Object van laag: {g} gevonden in een radius van 10 km, download van service...'.format(g=x.text))
                    outfile=x.text.replace('-','')
                    with open(os.path.join(os.path.split(outFC)[0],outfile+".json"), "wb") as ms_json:
                        ms_json.write(ft.content)
                    arcpy.conversion.JSONToFeatures(outfile+".json", os.path.join(outFC,outfile))
                    count+=1
                    analist.append(outfile)
                else:
                    pass
        if count==0:
            arcpy.AddMessage('Bedankt voor het uitvoeren van de SHRA tool. Voor de huidge locatie zijn er geen bijzonderheden gevonden.')
        return(count,analist)

    def intersector(self,outFC,analist):
        for fn in analist:  
            out_feature_class=os.path.join(outFC,fn+'_intersected')
            in_features=[os.path.join(outFC,fn),os.path.join(outFC,'WellsAOI')]
            arcpy.analysis.Intersect(in_features, out_feature_class)
            ##if arcpy.management.GetCount(out_feature_class)[0] != '0':
                ##arcpy.AddMessage('{} was found in AOI of geothermal system'.format(fn))
                

            
    def deletewrapper(self,fc):
        arcpy.Delete_management(fc)

    def mapwrapper(self,mapname='Not filled'):
        aprx = arcpy.mp.ArcGISProject("Current")
        if mapname == 'Not filled':
            aprx.listMaps()[0].name='SRHA Analysis'
        else:
            count=0
            for Maps in aprx.listMaps():
                if Maps.name==mapname:
                    count+=1
            if count >0:
                pass
            else:
                aprx.listMaps()[0].name='SRHA Analysis'
        return(aprx.listMaps()[0].name)

    def resultmessages(self):
        from collections import defaultdict
        layertocode = {
               1:["SHRA_geothermie_overzichtskaart:Seismisch_actief_gebied_Roerdalslenk","SHRA_geothermie_overzichtskaart:Seismisch_actief_gebied_Roerdalslenk","SHRA_geothermie_overzichtskaart:Gasvelden_Invloedsgebied","SHRA_geothermie_overzichtskaart:Gasvelden"],
               2:["SHRA_geothermie_overzichtskaart:Zoutvelden","SHRA_geothermie_overzichtskaart:Zoutvelden_Invloedsgebied"],
               3:["SHRA_geothermie_overzichtskaart:Relevante_breuken_Boven-Noordzee_Groep","SHRA_geothermie_overzichtskaart:Relevante_breuken_Rijnland_Groep","SHRA_geothermie_overzichtskaart:Relevante_breuken_Onder_Germaanse_Trias_Groep","SHRA_geothermie_overzichtskaart:Relevante_breuken_Zechstein_groep"]
        }

        codemessages = {
                    1:["Ga naar de volgende vraag in de beslisboom"],
                    2:["Ga naar SDRA Maatwerk"],
                    3:["Ga naar SDRA Standaard methode"]
            }

        outdict = defaultdict(list)

        for d in (layertocode,codemessages): # you can list as many input dicts as you want here
            for key, value in d.items():
                for values in value:
                    outdict[key].append(values)

        return(outdict)
    
    def execute(self, params, messages):

        arcpy.env.addOutputsToMap=False
        arcpy.env.overwriteOutput = True
        arcpy.mp.ArcGISProject("Current").listMaps()[0].spatialReference = params[3].value
        spatialrefrence=arcpy.SpatialReference('GCS_WGS_1984')
        arcpy.AddMessage('SHRA tool gebruikt EPSG: {} CRS voor de output:'.format(params[3].value.factoryCode))
              

        
        outFC=params[2].valueAsText
        #if spatialrefrence.factoryCode not in(4326,23031,28992,3857):
         #   self.errorhandler(1000)
        injw=params[0].values
        iwg=[]
        # creating points for injection wells
        for iw in injw:
            coords = str(iw).replace(',','.').split(' ')

            x = float(coords[0])
            y = float(coords[1])
            arcpy.AddMessage('SHRA tool gebruikt de volgende coordinaten voor de injectie put: {x},{y}'.format(x=x,y=y))   
            iwg.append([self.wellscreator(x,y,spatialrefrence),'injection'])

        prdw=params[1].valueAsText
        pwg=[]
        # creating points for production wells
      ##  for pw in prdw:
        coords = str(prdw).replace(',','.').split(' ')

        x = float(coords[0])
        y = float(coords[1])
        arcpy.AddMessage('SHRA tool gebruikt de volgende coordinaten voor de injectie put: {x},{y}'.format(x=x,y=y))
        pwg.append([self.wellscreator(x,y,spatialrefrence),'production'])

        # check for distance between wells
        for injectors in iwg:
            for producers in pwg: # not needed anymore as production well can only be 1 for now
                angle,distance = producers[0].angleAndDistanceTo(injectors[0],'GEODESIC')
                injectors.append(distance)
                arcpy.AddMessage('Putten liggen {} meter uit elkaar'.format(distance))
        # creating well features
        wells=pwg+iwg
        
        
        self.featurecreator(wells,outFC,spatialrefrence,'POINT','Wells')
        wells=self.buffer(wells,os.path.join(outFC,'Wells'),outFC)
        self.featurecreator(wells,outFC,spatialrefrence,'POLYGON','Wellsbuffered')
        # creating area of influence well
        
        self.convexhull(os.path.join(outFC,'Wellsbuffered'),os.path.join(outFC,'WellsAOI'),outFC)   
        bbox= self.boundingbox(os.path.join(outFC,'WellsAOI'),outFC)

        # getting relevant layers based on surrounding
        count,analist=self.serviceloader(bbox,outFC)
        self.intersector(outFC,analist)

        # Building rapport 
        mess=self.resultmessages()
        # building map
        mapname=params[4].value
        mapname=self.mapwrapper(mapname)
        self.mapbuilder(mapname,outFC,analist,mess)
        """The source code of the tool."""



